%% Generate experimental sequence
clear design
design_raw = lhsdesign(29,4);
design = round(design_raw,1)
cat_high = 0.01; %M
act_high = 0.01; %M
mon_high = 1; % M

design (:,1) = design(:,1) .* cat_high;
design (:,2) = design(:,2) .* act_high;
design (:,3) = design(:,3) .* mon_high;

design

%% Initiate data analysis
close all
clc
clearvars -except
font_size = 70;
raw_data_import = xlsread ('Results from all trials.xlsx');

%% Find exotherm and asign constants
clear conversion peakValues peak_locations experimental_zones average_exotherm std_exotherm conversion total_conversion plot_vec conversionvec

[number_of_points, number_of_zones] = size(raw_data_import);
for i = 1:number_of_points
    exotherm(i) = raw_data_import(i,21) - min(min(raw_data_import));
end
[peakValues, peak_locations] = findpeaks(exotherm(1:4531),'MinPeakProminence',0.9);

[peakValues, peak_locations(30:37)] = (findpeaks(raw_data_import([4531:end],32),'MinPeakProminence',0.9));
peak_locations(30:37)= peak_locations(30:37) + 20 + 4531; % Move to next experiment, add number of points before temp trials

conc_cat = [0.006 0.004 0.009 0.002 0.001 0.009 0.007 0.007 0.004 0.005 0.003 0.002 0.009 0.003 0.006 0.001 0.005 0.003 0.008 0.005 0.007 0.003 0.002 0.001 0.01 0.009 0.007 0.004 0.001 0.001 0.001 0.001 0.001 0.001 0.001 0.001 0.001];
conc_act = [0.005 0.001 0.01 0.001 0.006 0.008 0.004 0.003 0.005 0.003 0.004 0.002 0.006 0.003 0.002 0.009 0.008 0.009 0.001 0.006 0.002 0.008 0.006 0.007 0.007 0.001 0.005 0.01 0.003 0.009 0.009 0.009 0.009 0.009 0.009 0.009 0.009];
conc_mon = [0.3 1 0.6 1 0.1 0.2 0.5 0.7 0.4 0.7 0.4 0.9 0.6 0.7 0.3 0.1 0.1 0.9 0.8 0.8 0.4 0.1 0.2 0.3 0.1 0.9 0.2 0.7 0.6 1 1 1 1 1 1 1 1];
feed_temp = [22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 22.9 15.63 10.73 8.43 6.31 41.88 46.1 60.72 78.78];

FR(1:29) = 60 * 60; %mL/hr
FR(30:37) = 20 * 60; %mL/hr
reactor_volume = 49.483E-3; %L

DeltaHrx = -266E3; %J/mol

HC_chlbnz = 150.1; %J/(mol*K)
HC_hex = 183.3; %J/(mol*K)
HC_tol = 155.96; %J/(mol*K)
HC_poly = 50;

MM_cat = 448.53; %g/mol
MM_act = 511.98; %g/mol
MM_chlbnz = 112.56; %g/mol
MM_hex = 84.1608; %g/mol
MM_tol = 92.14; %g/mol

den_tol = 0.87; %g/mL
den_chlbnz = 1.106; %g/mL

RT = reactor_volume ./ (FR); % Residence time

MF_hex_exp = conc_mon .* (FR); % mol/sec

MF_chlbnz = (FR .* den_tol) ./ MM_tol;
MF_tol = (FR .* den_chlbnz) ./ MM_chlbnz;

for i = 1:length (peak_locations)
    experimental_zones(i,1) = peak_locations(i)-20;
    experimental_zones(i,2) = peak_locations(i);
end

for i = 1:length (peak_locations)
    average_exotherm(i) = mean (exotherm(experimental_zones(i,1):experimental_zones(i,2)));
    std_exotherm(i) = std (exotherm(experimental_zones(i,1):experimental_zones(i,2)));
    clear Tvec
    
    for j = 1:10
        Tvec(j) = abs(raw_data_import(peak_locations(i),22-j) - feed_temp(i));
    end
    
    conversionvec(i,:) = conversionscript(MF_tol(i),MF_chlbnz(i),MF_hex_exp(i),Tvec);
    
    total_conversion(i) = sum(conversionvec(i,:))*100;
end

% Calculate and plot catalytic activity
cat_activity = (conc_mon .* total_conversion) ./ (conc_cat .* conc_mon .* RT);
plot (cat_activity,'o')

%% Train different neural networks
clear perf mse inputs targets

inputs = [conc_cat; conc_mon; conc_act; feed_temp; FR]; % Assign network inputs
targets = [cat_activity]/max(cat_activity); % Assign network targets and scale to 0-1

reps = 5; % Number of averaging repetitions for network performance comparison

%Train the networks
for g = 9
    switch g % Different training algorithms
        case 1
            trainFcn = 'trainlm';
        case 2
            trainFcn = 'trainbfg';
        case 3
            trainFcn = 'trainrp';
        case 4
            trainFcn = 'trainscg';
        case 5
            trainFcn = 'traincgb';
        case 6
            trainFcn = 'traincgf';
        case 7
            trainFcn = 'traincgp';
        case 8
            trainFcn = 'trainoss';
        case 9
            trainFcn = 'trainbr';
    end
    for i=1:20 % Different number of neurons
        temp_performance_var = 100;
        for repvar = 1:reps
            hiddenLayerSize = i;
            net = fitnet(hiddenLayerSize,trainFcn);
            net.divideParam.trainRatio = 70/100;
            net.divideParam.valRatio = 15/100;
            net.divideParam.testRatio = 15/100;
            [net,tr] = train(net,inputs,targets); % Train network
            temp_performance_var (repvar) = abs(mean((targets - (net(inputs))) ./ targets) * 100); % Calculate performance (error %)
            ind = tr.testInd;
            temp_performance_var_test = abs(mean((targets(ind) - (net(inputs(:,ind)))) ./ targets(ind)) * 100);
        end
        net_performance (g,i) = mean (temp_performance_var)
        net_performance_test (g,i) = mean (temp_performance_var_test)
    end
end

%% Error error percentage heatmap
% Plot normalized error matrix heatmap
clear error_matrix
error_matrix_test = net_performance_test;
imagesc(error_matrix_test(:,1:20))
hold on
[C,h] = contour (error_matrix_test)
h.LineWidth = 2;
h.LineColor = 'k';
caxis([0 50])
colorbar
ylabel('Training method')
xlabel('Number of hidden layer neurons')
c = colorbar;
c.Label.String='Percent error'
set(gca,'FontSize',font_size,'fontWeight','bold','lineWidth',5)
set(gca, 'box', 'off')
set(gca,'TickDir','out');

%% Loss over different test splits

net2 = fitnet(3,'trainbr');
net2.divideParam.trainRatio = 70/100;
net2.divideParam.valRatio = 20/100;
net2.divideParam.testRatio = 10/100;
[net2,tr] = train(net2,inputs,targets); % Train network
            
ind = tr.testInd
test_loss = abs(mean((targets(ind) - (net2(inputs(:,ind)))) ./ targets(ind)) * 100)

%% Make performance plot

tr = network_used_training;

plotperform(tr)
set(gca,'FontSize',font_size,'fontWeight','bold','lineWidth',5)
set(gca,'TickDir','out');
set(gca, 'box', 'off')

%% Error individual plots
for i = 1:9
    subplot (5,2,i)
    semilogy (net_performance(i,:),'o')
    title ([num2str(mean(net_performance(i,:))), ', ', num2str(min(net_performance(i,:)))])
    axis ([0 20 0 200])
    xlabel ('Number of hidden layer neurons')
    ylabel ('Average error %')
    set(gca,'FontSize',font_size/2,'fontWeight','bold')
    set(gca,'TickDir','out');
end

%% Train final network
clear perf mse inputs targets
inputs = [conc_cat; conc_mon; conc_act; feed_temp; FR]; % Nerual network inputs
targets = cat_activity/max(cat_activity); % Neural network targers

hiddenLayerSize = 9; % Number of hidden layer neurons
trainFcn = 'trainbr'; % Training function

percent_error_temp = 100;

while percent_error_temp > 0.5 % Set desired error percent
    net = fitnet(hiddenLayerSize,trainFcn);
    %net.Layers{2}.transferFcn='poslin';
    net.divideParam.trainRatio = 100/100;
    net.divideParam.valRatio = 15/100;
    net.divideParam.testRatio = 15/100;
    [net,tr] = train(net,inputs,targets);
    percent_error_temp = abs(mean((targets - (net(inputs))) ./ targets) * 100)
end

%% View Percent error over test indecies
ind = tr.testInd;
percent_error_test_ind = abs(mean((targets(ind) - (net(inputs(:,ind)))) ./ targets(ind)) * 100)

%% Compare AI and experimental data
opt = 2; % 1 plots points, 2 plots bars with standard deviation

if opt == 1
    comparison_dataset = cat_activity;
    plot (comparison_dataset,'r*','markersize',20)
end
if opt == 2
    plot_vec = [1:1:37];
    error_std = std_exotherm./average_exotherm;
    plot_vec = [1:1:length(average_exotherm)]
    bar(plot_vec,cat_activity)
    hold on
    er = errorbar(plot_vec,cat_activity,cat_activity .* error_std/2,cat_activity .* error_std/2);
    er.LineWidth = 10;
    er.LineStyle = 'none';
    er.CapSize = 50;
end

hold on
AI_results = abs(net(inputs)*max(cat_activity));
plot (AI_results,'bo','markersize',20,'MarkerFaceColor','r')
xlabel ('Experiment number')
ylabel ('Catalytic productivity (g pol/([mon]*[cat]*hr)')
legend ('Experimental data', 'Standard Deviation', 'AI prediction')
title (['Average percent error = ', num2str(round(percent_error_temp,3)), '%'])
set(gca,'FontSize',font_size,'fontWeight','bold','lineWidth',5)
set(gca, 'box', 'off')
set(gca,'TickDir','out');

%% Perform AI experimental space mapping
clear AI_output
number_of_divisions = 100;

range_cat = linspace(min(conc_cat),max(conc_cat),number_of_divisions)
range_mon = linspace(min(conc_mon),max(conc_mon),number_of_divisions)
range_act = linspace(min(conc_act),max(conc_act),number_of_divisions)

itt = 1;
totalitt = length (range_cat) * length (range_mon) * length (range_act);

f = waitbar (itt/totalitt, ['Calculation' num2str(itt/totalitt) '% complete'] )
for ind1 = 1:length (range_cat)
    for ind2 = 1:length (range_mon)
        for ind3 = 1:length (range_act)
            inputset = [range_cat(ind1);range_mon(ind2);range_act(ind3);20;FR(1)];
            AI_output(ind1,ind2,ind3) = abs(net(inputset))*max(cat_activity);
            itt = itt+1;
        end
        waitbar (itt/totalitt, f, ['Calculation' num2str(round(itt*100/totalitt)) '% complete'])
    end
end

%% Plot AI results
plot_num = 1;
for j = 1:3
    for i = round(linspace(1,number_of_divisions,3))
        subplot (3,3,plot_num)
        switch j
            case 1
                surfc(range_cat,range_mon,squeeze(log(AI_output(:,:,i))),'edgecolor','flat','FaceAlpha',0.5)
                title (['Activator concentration ' num2str(round(range_act(i),3)) ' M'])
                xlabel ('[Catalyst], mol/L')
                ylabel ('[Monomer], mol/L')
                zlabel('Log of Catalytic Productivity')
            case 2
                surfc(range_cat,range_act,squeeze(log(AI_output(:,i,:))),'edgecolor','flat','FaceAlpha',0.5)
                title (['Monomer concentration ' num2str(round(range_mon(i),3)) ' M'])
                xlabel ('[Catalyst], mol/L')
                ylabel ('[Activator], mol/L')
                zlabel('Log of Catalytic Productivity')
            case 3
                surfc(range_act,range_mon,squeeze(log(AI_output(i,:,:))),'edgecolor','flat','FaceAlpha',0.5)
                title (['Catalyst concentration ' num2str(round(range_cat(i),3)) ' M'])
                xlabel ('[Activator], mol/L')
                ylabel ('[Monomer], mol/L')
                zlabel('Log of Catalytic Productivity')
        end
        %xlim ([4E-6 6E-6])
        %zlim ([1E4 10E6])
        set(gca,'FontSize',font_size/2,'fontWeight','bold','lineWidth',5)
        set(gca,'TickDir','out');
        set(gca, 'box', 'off')
        view (25, 10)
        plot_num = plot_num+1;
    end
end

%% AI for temp range
number_of_points = 1000;

conc_cat_min = min (conc_cat);
conc_cat_max = max (conc_cat);
conc_mon_min = min (conc_mon)/2;
conc_mon_max = max (conc_mon)*2;
conc_act_min = min (conc_act);
conc_act_max = max (conc_act);


conc_cat_range=sort(conc_cat_min+rand(1,number_of_points)*(conc_cat_max-conc_cat_min));
conc_mon_range=sort(conc_mon_min+rand(1,number_of_points)*(conc_mon_max-conc_mon_min));
conc_act_range=sort(conc_act_min+rand(1,number_of_points)*(conc_act_max-conc_act_min));
temp_range = linspace(min(feed_temp), max(feed_temp), number_of_points);

temp_dependant_output = [];


for k = 1:number_of_points
    for i = 1:length (temp_range)
        temp_dependant_output(k,i) = abs(net([conc_cat_range(k); conc_mon_range(k); conc_act_range(k); temp_range(i); FR(1)]))*max(cat_activity);
    end
    k
end

%% Make temperature range plot
imagesc (conc_mon_range, temp_range, temp_dependant_output/10^7)

xlabel ('Monomer Concentration, M')
ylabel ('Temperature, C')
cb = colorbar;
cb.Label.String = 'Catalytic productivity (x10^7)'
set(gca,'FontSize',font_size/1.5,'fontWeight','bold','lineWidth',5)
set(gca,'TickDir','out');
set(gca, 'box', 'off')

%% Plot experimental sequence
exp_vec = [1:1:29];
yyaxis left
hold on
plot (exp_vec,conc_cat(1:29),'ro','MarkerSize',50,'MarkerFaceColor', 'r')
plot (exp_vec,conc_act(1:29),'bs','MarkerSize',50,'MarkerFaceColor', 'b')
axis ([0 30 0 0.011])
ylabel ('Concentration catalyst and activator, M')
set(gca,'ycolor','k') 
yyaxis right
plot (exp_vec,conc_mon(1:29),'gd','MarkerSize',50,'MarkerFaceColor', 'g')
axis ([0 30 0 1.1])
ylabel ('Concentration monomer, M')
set(gca,'ycolor','k') 
xlabel ('Experiment number')

set(gca,'FontSize',font_size,'fontWeight','bold','lineWidth',5)
set(gca,'TickDir','out');
set(gca, 'box', 'off')
%legend ('Catalyst concentration', 'Activator concentration', 'Monomer concentration')

%% Kinetic rate constants
subplot (2,1,1)
ax = gca;
imagesc(log(kp), 'Parent', ax);
ax.YAxis.Visible = 'off';
xlabel ('Experiment number')
cb = colorbar;
cb.Label.String = 'log(k_p), M^{-1} s^{-1}'
set(gca,'FontSize',font_size/2,'fontWeight','bold','lineWidth',5)
set(gca,'TickDir','out');
set(gca, 'box', 'off')

subplot (2,1,2)
ax = gca;
imagesc(log(ki), 'Parent', ax);
ax.YAxis.Visible = 'off';
xlabel ('Experiment number')
cb = colorbar;
cb.Label.String = 'log(k_i), M^{-1} s^{-1}'
set(gca,'FontSize',font_size/2,'fontWeight','bold','lineWidth',5)
set(gca,'TickDir','out');
set(gca, 'box', 'off')

%% From rate constants code, concentration versus time plots
volume_plot_vec = Vspan / max (Vspan);
hold on
for i = 1:29
    plot(volume_plot_vec,Conc_Hex(i,:),'LineWidth',15)
end
xlabel('Dimensionless reactor volume')
ylabel('Concentration hexene [M]');
%legend('Test 1','Test 2','Test 3','Test 4','Test 5','Test 6','Test 7')
set(gca,'FontSize',font_size,'fontWeight','bold')
set(gca,'TickDir','out'); % The only other option is 'in'

%% Optimize output
min_temp = 10;
max_temp = 75;

% From Simga-Aldirch, February 2020
cat_unit_cost = 1000/MM_cat; % $/mol
act_unit_cost = 168/MM_act; % $/mol
mon_unit_cost = 62*0.673/MM_hex; % $/mol
example_price_polymer = 0.1; % $/gram

cost_cat = conc_cat_range * cat_unit_cost;
cost_act = conc_act_range * act_unit_cost;
cost_mon = conc_mon_range * mon_unit_cost;

polymer_profit = (temp_dependant_output * example_price_polymer .* conc_cat_range .* conc_mon_range)...
    - (cost_cat + cost_act + cost_mon);
max_profit = max(max(polymer_profit)); % Max proffit / mL / hr
optimal_index = find (polymer_profit == max_profit);
array_dimensions = [number_of_points, temp_range];
[opt_concs, opt_temp] = ind2sub (array_dimensions,optimal_index);

optimal_cat_concentration = conc_cat_range(opt_concs)
optimal_act_concentration = conc_act_range(opt_concs)
optimal_mon_concentration = conc_mon_range(opt_concs)
optimal_temperature = temp_range(opt_temp)

%% Digital twim estimation
clear design_for_DT design_DT inputs_for_DT outputs_from_DT net_for_DT AI_output_DT optimal_cat_concentration_DT optimal_act_concentration_DT...
    optimal_act_concentration_DT optimal_mon_concentration_DT optimal_temperature_DT net_for_DT gradient max_activity

% From Simga-Aldirch, February 2020
cat_unit_cost = 1000/MM_cat; % $/mol
act_unit_cost = 168/MM_act; % $/mol
mon_unit_cost = 62*0.673/MM_hex; % $/mol
example_price_polymer = 0.1; % $/gram

max_itterations = 500;
for DT_itteration = 1:max_itterations
    clear design_for_DT design_DT outputs_from_DT
    
    if DT_itteration == 1
        opt_routine = 1;
    elseif DT_itteration > 3
        opt_routine = 2;
    end
    
switch opt_routine
    case 1
        number_of_points_DT = 100;
        cat_high_DT = max(conc_cat);
        cat_low_DT = min(conc_cat); %M
        act_high_DT = max(conc_act); %M
        act_low_DT = min(conc_act); %M
        mon_high_DT = 2.5; % M
        mon_low_DT = 0.01; % M
        temp_high_DT = 80;
        temp_low_DT = 30;
        
    case 2
        if DT_itteration < 15
            learning_rate = 2;
        elseif DT_itteration > 15
            learning_rate = search_gradient*1.5;
            if learning_rate < 1
                learning_rate = 1.09;
            elseif learning_rate > 2;
                learning_rate = 2;
            end
        end
        
        max_productivity_value = max(max_activity);
        max_productivity_index = find (max_activity == max_productivity_value);

        number_of_points_DT = 30;
        cat_high_DT_temp = cat_high_DT;
        cat_high_DT = optimal_cat_concentration_DT(max_productivity_index) * learning_rate; %M
        if cat_high_DT > 0.01
            cat_high_DT = 0.009;
        end
        
        cat_low_DT_temp = cat_low_DT;
        cat_low_DT = optimal_cat_concentration_DT(max_productivity_index) / learning_rate; %M
        if cat_low_DT < 0.01 / 1000
            cat_low_DT = 0.01 / 1000;
        end
        
        act_high_DT_temp = act_high_DT;
        act_high_DT = optimal_act_concentration_DT(max_productivity_index) * learning_rate; %M
        if act_high_DT > 0.01
            act_high_DT = 0.01;
        end
        act_low_DT_temp = act_low_DT;
        act_low_DT = optimal_act_concentration_DT(max_productivity_index) / learning_rate; %M
        if act_low_DT < 0.01 / 1000
            act_low_DT = 0.01 / 1000;
        end
        
        mon_high_DT_temp = mon_high_DT;
        mon_high_DT = optimal_mon_concentration_DT(max_productivity_index) * learning_rate; % M
        if mon_high_DT > 2.5
            cat_high_DT = 2.5;
        end
        mon_low_DT_temp = mon_low_DT;
        mon_low_DT = optimal_mon_concentration_DT(max_productivity_index) / learning_rate; % M
        if mon_low_DT < 0.01
            mon_low_DT = 0.01;
        end
        
        temp_high_DT = optimal_temperature_DT(max_productivity_index) * learning_rate;
        if temp_high_DT > 80
            temp_high_DT = 80;
        end
        
        temp_low_DT = optimal_temperature_DT(max_productivity_index) / learning_rate;
        if temp_low_DT < 5
            temp_low_DT = 5;
        end
end


design_for_DT = lhsdesign(number_of_points_DT,4);
design_DT = round(design_for_DT,1);

design_DT (:,1) = design_DT(:,1) * (cat_high_DT-cat_low_DT) + cat_low_DT;
design_DT (:,2) = design_DT(:,2) * (mon_high_DT-mon_low_DT) + mon_low_DT;
design_DT (:,3) = design_DT(:,3) * (act_high_DT-act_low_DT) + act_low_DT;
design_DT (:,4) = design_DT(:,4) * (temp_high_DT-temp_low_DT) + temp_low_DT;
design_DT (:,5) = 3600;

inputs_for_DT = design_DT;

for i = 1:length(inputs_for_DT)
    input_set = [design_DT(i,1);design_DT(i,2);design_DT(i,3);design_DT(i,4);design_DT(i,5)];
    outputs_from_DT(i) = abs(net(input_set))*normalization_factor;
end

clear net_for_DT

net_for_DT= fitnet(9,'trainbr');
net_for_DT.divideParam.trainRatio = 70/100;
net_for_DT.divideParam.valRatio = 15/100;
net_for_DT.divideParam.testRatio = 15/100;

perf_temp = 100;

while perf_temp > 5
    [net_for_DT,tr_for_DT] = train(net_for_DT,design_DT',outputs_from_DT);
    ind = tr_for_DT.testInd;
    perf_temp = abs(mean((outputs_from_DT(ind) - (net_for_DT(design_DT(ind,:)'))) ./ outputs_from_DT(ind)) * 100)
end

number_of_divisions = 3;

range_cat_DT = linspace(cat_low_DT,cat_high_DT,number_of_divisions);
range_act_DT = linspace(act_low_DT,act_high_DT,number_of_divisions);
range_mon_DT = linspace(mon_low_DT,mon_high_DT,number_of_divisions);
range_temp_DT = linspace(temp_low_DT,temp_high_DT,number_of_divisions);

itt = 1;
totalitt = length (range_cat) * length (range_mon) * length (range_act);

for ind1 = 1:length (range_cat_DT)
    for ind2 = 1:length (range_mon_DT)
        for ind3 = 1:length (range_act_DT)
            for ind4 = 1:length (range_temp_DT)
                inputset = [range_cat_DT(ind1);range_mon_DT(ind2);range_act_DT(ind3);range_temp_DT(ind4);3600];
                AI_output_DT(ind1,ind2,ind3,ind4) = abs(net_for_DT(inputset));
                itt = itt+1;
            end
        end
    end
end

cost_cat_DT = range_cat_DT * cat_unit_cost;
cost_act_DT = range_act_DT * act_unit_cost;
cost_mon_DT = range_mon_DT * mon_unit_cost;

polymer_profit_DT = (AI_output_DT * example_price_polymer .* range_cat_DT .* range_act_DT)...
    - (cost_cat_DT + cost_act_DT + cost_mon_DT);

max_cat_profit_DT = max(max(max(max(polymer_profit_DT))));
max_cat_activity = max(max(max(max(AI_output_DT))));
optimal_index_DT = find (AI_output_DT == max_cat_activity);

array_dimensions_DT = [number_of_divisions; number_of_divisions; number_of_divisions; number_of_divisions];
[opt_cat_DT, opt_mon_DT, opt_act_DT, opt_temp_DT] = ind2sub (array_dimensions_DT,optimal_index_DT);

max_activity(DT_itteration) = max_cat_activity;

max_profit(DT_itteration) = max(max(max(max(polymer_profit_DT))));
optimal_cat_concentration_DT(DT_itteration) = range_cat_DT(opt_cat_DT(1));
optimal_act_concentration_DT(DT_itteration) = range_act_DT(opt_act_DT(1));
optimal_mon_concentration_DT(DT_itteration) = range_mon_DT(opt_mon_DT(1));
optimal_temperature_DT(DT_itteration) = range_temp_DT(opt_temp_DT(1));
learning_rate_plot(DT_itteration) = learning_rate;

DT_itteration = DT_itteration + 1
subplot (3,1,1)
yyaxis left
hold on
plot ((optimal_cat_concentration_DT),'r','LineWidth',15)
plot ((optimal_act_concentration_DT),'b','LineWidth',15)
ylabel ('Concentration cat. & act., M')
set(gca,'ycolor','k') 
yyaxis right
plot ((optimal_mon_concentration_DT),'g','LineWidth',15)
ylabel ('Concentration monomer, M')
set(gca,'ycolor','k')
xlabel ('Iteration number')
legend ('Catalyst concentration', 'Activator concentration', 'Monomer concentration')
set(gca,'FontSize',font_size/2,'fontWeight','bold')
set(gca,'TickDir','out');

subplot (3,1,2)
plot (optimal_temperature_DT,'LineWidth',15)
ylabel ('Temperature, C')
xlabel ('Iteration number')
set(gca,'FontSize',font_size/2,'fontWeight','bold')
set(gca,'TickDir','out');

if DT_itteration > 15
    search_gradient = mean(abs(diff(max_profit(end-5:end))/max(diff(max_profit(end-5:end)))))
end

end

%% Plot digital twin results
subplot (1,2,1)
yyaxis left
hold on
plot ((optimal_cat_concentration_DT),'-r','LineWidth',15)
plot ((optimal_act_concentration_DT),'-b','LineWidth',15)
ylabel ('Concentration cat. & act., M')
set(gca,'ycolor','k') 
yyaxis right
plot ((optimal_mon_concentration_DT),'-g','LineWidth',15)
ylabel ('Concentration monomer, M')
set(gca,'ycolor','k')
xlabel ('Iteration number')
legend ('Catalyst concentration', 'Activator concentration', 'Monomer concentration')
set(gca,'FontSize',font_size,'fontWeight','bold')
set(gca,'TickDir','out');

subplot (1,2,2)
plot (optimal_temperature_DT(1:50),'LineWidth',15)
ylabel ('Temperature, C')
xlabel ('Iteration number')
set(gca,'FontSize',font_size,'fontWeight','bold')
set(gca,'TickDir','out');

%% Create annimation
clear v M annimationindex_act annimationindex_mon annimationindex_cat
for j = 1:3
    
    if j == 1
        for annimationindex_act = 1:length(range_act)
            surfc(range_cat,range_mon,squeeze(AI_output(:,:,annimationindex_act)))
            set(gcf, 'Units', 'Normalized', 'OuterPosition', [0 0 1 1]);
            xlabel ('Catalyst concentration')
            ylabel ('Monomer concentration')
            zlabel ('Catalyst activity')
            title (['Activator concentration ' num2str(range_act(annimationindex_act)) ' Molar'])
            set(gca,'FontSize',font_size,'fontWeight','bold')
            drawnow
            M(annimationindex_act) = getframe(gcf);
        end
        video_title = 'Activator Range.avi'
        v = VideoWriter(video_title);
        v.FrameRate = 10;
        open(v);
        writeVideo(v,M)
        close(v);
    end
    
    if j == 2
        clear v M
        for annimationindex_mon = 1:length(range_mon)
            surfc(range_cat,range_mon,squeeze(AI_output(:,annimationindex_mon,:)))
            set(gcf, 'Units', 'Normalized', 'OuterPosition', [0 0 1 1]);
            xlabel ('Catalyst concentration')
            ylabel ('Activator concentration')
            zlabel ('Catalyst activity')
            title (['Monomer concentration ' num2str(range_mon(annimationindex_mon)) ' Molar'])
            set(gca,'FontSize',font_size,'fontWeight','bold')
            %     set(get(gca,'xlabel'),'rotation',18)
            %     set(get(gca,'ylabel'),'rotation',-20)
            drawnow
            M(annimationindex_mon) = getframe(gcf);
        end
        video_title = 'Monomer Range.avi'
        v = VideoWriter(video_title);
        v.FrameRate = 10;
        open(v);
        writeVideo(v,M)
        close(v);
    end
    
    if j == 3
        clear v M
        for annimationindex_cat = 1:length(range_cat)
            surfc(range_cat,range_mon,squeeze(AI_output(annimationindex_cat,:,:)))
            set(gcf, 'Units', 'Normalized', 'OuterPosition', [0 0 1 1]);
            xlabel ('Monomer concentration')
            ylabel ('Activator concentration')
            zlabel ('Catalyst activity')
            title (['Catalyst concentration ' num2str(range_cat(annimationindex_cat)) ' Molar'])
            set(gca,'FontSize',font_size,'fontWeight','bold')
            %     set(get(gca,'xlabel'),'rotation',18)
            %     set(get(gca,'ylabel'),'rotation',-20)
            drawnow
            M(annimationindex_cat) = getframe(gcf);
        end
        video_title = 'Catalyst Range.avi'
        v = VideoWriter(video_title);
        v.FrameRate = 10;
        open(v);
        writeVideo(v,M)
        close(v);
    end
end